using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;

namespace AspNetCoreUploadFileExample.Extensions
{
    public static class FileHelpers
    {
        /// <summary>
        /// For more file signatures, see the File Signatures Database and the official specifications 
        /// for the file types you wish to add.
        /// </summary>
        private static readonly Dictionary<string, List<byte[]>> _fileSignature = new Dictionary<string, List<byte[]>>
        {
            { ".gif", new List<byte[]> { new byte[] { 0x47, 0x49, 0x46, 0x38 } } },
            { ".png", new List<byte[]> { new byte[] { 0x89, 0x50, 0x4E, 0x47, 0x0D, 0x0A, 0x1A, 0x0A } } },
            { ".jpeg", new List<byte[]>
                {
                    new byte[] { 0xFF, 0xD8, 0xFF, 0xE0 },
                    new byte[] { 0xFF, 0xD8, 0xFF, 0xE2 },
                    new byte[] { 0xFF, 0xD8, 0xFF, 0xE3 },
                }
            },
            { ".jpg", new List<byte[]>
                {
                    new byte[] { 0xFF, 0xD8, 0xFF, 0xE0 },
                    new byte[] { 0xFF, 0xD8, 0xFF, 0xE1 },
                    new byte[] { 0xFF, 0xD8, 0xFF, 0xE8 },
                }
            },
            { ".zip", new List<byte[]>
                {
                    new byte[] { 0x50, 0x4B, 0x03, 0x04 },
                    new byte[] { 0x50, 0x4B, 0x4C, 0x49, 0x54, 0x45 },
                    new byte[] { 0x50, 0x4B, 0x53, 0x70, 0x58 },
                    new byte[] { 0x50, 0x4B, 0x05, 0x06 },
                    new byte[] { 0x50, 0x4B, 0x07, 0x08 },
                    new byte[] { 0x57, 0x69, 0x6E, 0x5A, 0x69, 0x70 },
                }
            },
        };

        /// <summary>
        /// A file's signature is determined by the first few bytes at the start of a file. These bytes can be used to 
        /// indicate if the extension matches the content of the file. The sample app checks file signatures for a few 
        /// common file types. In the following example, the file signature for a JPEG image is checked against the file:
        /// https://docs.microsoft.com/en-us/aspnet/core/mvc/models/file-uploads
        /// </summary>
        public static bool IsValidFileExtensionAndSignature(string fileName, Stream dataStream, string[] permittedExtensions)
        {
            if (string.IsNullOrEmpty(fileName) || dataStream == null || dataStream.Length == 0)
            {
                return false;
            }

            var ext = Path.GetExtension(fileName).ToLowerInvariant();

            if (string.IsNullOrEmpty(ext) || !permittedExtensions.Contains(ext))
            {
                return false;
            }

            dataStream.Position = 0;

            using var binaryReader = new BinaryReader(dataStream);

            var signatures = _fileSignature[ext];

            var headerBytes = binaryReader.ReadBytes(signatures.Max(m => m.Length));

            return signatures.Any(signature => headerBytes.Take(signature.Length).SequenceEqual(signature));
        }
    }
}
