﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Threading.Tasks;
using AspNetCoreUploadFileExample.Extensions;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Http.Extensions;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.ModelBinding;
using Microsoft.AspNetCore.WebUtilities;
using Microsoft.Net.Http.Headers;

namespace AspNetCoreUploadFileExample.Controllers
{
    public class FileController : Controller
    {
        private readonly IWebHostEnvironment _hostEnvironment;

        public FileController(IWebHostEnvironment hostEnvironment)
        {
            _hostEnvironment = hostEnvironment;
        }

        [HttpPost]
        [DisableFormValueModelBinding]
        public async Task<IActionResult> Upload()
        {
            KeyValueAccumulator formAccumulator = new KeyValueAccumulator();

            string multipartBoundary = Request.GetMultipartBoundary();

            if (string.IsNullOrWhiteSpace(multipartBoundary))
            {
                return BadRequest();
            }

            MultipartReader multipartReader = new MultipartReader(multipartBoundary, Request.Body);

            MultipartSection multipartSection = await multipartReader.ReadNextSectionAsync();

            while (multipartSection != null)
            {
                var fileMultipartSection = multipartSection.AsFileSection();

                if (fileMultipartSection != null)
                {
                    using MemoryStream memoryStream = new MemoryStream();

                    await fileMultipartSection.Section.Body.CopyToAsync(memoryStream);

                    // Check if the file is empty or exceeds the size limit.
                    if (memoryStream.Length == 0)
                    {
                        return Content("文件内容不能为空");
                    }

                    if (memoryStream.Length > 1024 * 1024 * 3)
                    {
                        return Content("文件不能大于3MB");
                    }

                    string fileName = fileMultipartSection.FileName;

                    var permittedExtensions = new string[] { ".jpg", ".png", ".jpeg" };

                    var mediaTypeHeaderValue = MediaTypeHeaderValue.Parse(multipartSection.ContentType);
                    string subTypeValue = mediaTypeHeaderValue.SubType.Value;
                    if (!permittedExtensions.Any(e => e.TrimStart('.') == subTypeValue.ToLowerInvariant()))
                    {
                        return Content("无效的文件格式");

                    }

                    if (!FileHelpers.IsValidFileExtensionAndSignature(fileName, memoryStream, permittedExtensions))
                    {
                        return Json(new ValidationProblemDetails { Title = "仅允许JPG和PNG格式" });

                    }

                    string path = Path.Combine(_hostEnvironment.ContentRootPath, "Uploads", fileName);

                    Directory.CreateDirectory(Path.GetDirectoryName(path));

                    using var fileStream = new FileStream(path, FileMode.Create);
                    await fileMultipartSection.FileStream.CopyToAsync(fileStream);
                }

                var formMultipartSection = multipartSection.AsFormDataSection();

                if (formMultipartSection != null)
                {
                    var value = await formMultipartSection.GetValueAsync();
                    formAccumulator.Append(formMultipartSection.Name, value);
                }

                multipartSection = await multipartReader.ReadNextSectionAsync();
            }

            var formCollection = new FormCollection(formAccumulator.GetResults());

            var formValueProvider = new FormValueProvider(BindingSource.Form, formCollection, CultureInfo.CurrentCulture);

            object myObjectModel = new object();

            await TryUpdateModelAsync(myObjectModel, prefix: string.Empty, valueProvider: formValueProvider);

            return View(myObjectModel);
        }
    }
}