﻿using BookStore.Books;
using Microsoft.EntityFrameworkCore.Internal;
using NSubstitute;
using Shouldly;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Volo.Abp.Application.Dtos;
using Volo.Abp.Validation;
using Xunit;

namespace BookStore.Samples
{
    public class BookAppServiceTests : BookStoreApplicationTestBase
    {
        private readonly IBookAppService _bookAppService;

        public BookAppServiceTests()
        {
            _bookAppService = GetRequiredService<IBookAppService>();
        }

        [Fact]
        public async Task Should_Get_List_Of_Books()
        {
            var result = await _bookAppService.GetListAsync(new PagedAndSortedResultRequestDto());

            result.TotalCount.ShouldBeGreaterThanOrEqualTo(1);

            result.Items.ShouldContain(b => b.Name == "Test book 1");
        }

        [Fact]
        public async Task Should_Create_A_Valid_Book()
        {
            var createUpdateBookDto = new CreateUpdateBookDto
            {
                Name = "Test book 3",
                Price = 10,
                PublishDate = DateTime.Now.AddDays(-1),
                Type = BookType.Biography
            };

            var result = await _bookAppService.CreateAsync(createUpdateBookDto);

            result.Id.ShouldNotBe(Guid.Empty);

            result.Name.ShouldBe("Test book 3");
        }


        [Fact]
        public async Task Should_Create_A_Without_Book()
        {
            var exception = await Assert.ThrowsAnyAsync<AbpValidationException>(async () =>
            {
                var createUpdateBookDto = new CreateUpdateBookDto
                {
                    Name = string.Empty,
                    Price = 10,
                    PublishDate = DateTime.Now.AddDays(-2),
                    Type = BookType.ScienceFiction
                };

                await _bookAppService.CreateAsync(createUpdateBookDto);
            });

            exception.ValidationErrors.ShouldContain(err => err.MemberNames.Any(m => m == "Name"));
        }

        [Fact]
        public async Task Should_Get_List_Of_Books_For_Substitute()
        {
            IBookAppService _bookAppService = Substitute.For<IBookAppService>();

            var bookDtoList = new List<BookDto>
            {
                new BookDto{ Id=Guid.NewGuid(), Name="EfCore", CreationTime=DateTime.Now },
                new BookDto{ Id=Guid.NewGuid(), Name="AspNetCore", CreationTime=DateTime.Now }
            };

            var bookDtoPagedList = new PagedResultDto<BookDto>(bookDtoList.Count, bookDtoList);

            var pagedRequestDto = new PagedAndSortedResultRequestDto();

            _bookAppService.GetListAsync(pagedRequestDto).Returns(Task.FromResult(bookDtoPagedList));

            var result = await _bookAppService.GetListAsync(pagedRequestDto);

            result.TotalCount.ShouldBe(2);
            result.Items.ShouldContain(b => b.Name == "AspNetCore");
        }
    }
}
