﻿using Microsoft.AspNetCore.Mvc;
using Moq;
using ProductWeb.Controllers;
using ProductWeb.DataAccess;
using ProductWeb.Services;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Xunit;

namespace ProductWeb.Tests
{
    public class MockProductControllerTest
    {
        public IEnumerable<Product> GetProducts()
        {
            return new List<Product> {
                new Product
                {
                    ProductID=10,
                    Count = 10,
                    Name = "Learning C++",
                    Description = "This is .NET Core blog."
                },
                new Product
                {
                    ProductID = 20,
                    Count=20,
                    Name = "Learning JAVA",
                    Description = "A best-selling book Java"
                }
            };
        }

        [Fact]
        public async Task List_ReturnViewResult_WithListOfProducts()
        {
            var mockProductService = new Mock<IProductService>();
            mockProductService.Setup(s=>s.ListAsync()).ReturnsAsync(GetProducts());
            var controller = new ProductController(mockProductService.Object);

            var result = await controller.List();

            var viewResult = Assert.IsType<ViewResult>(result);

            var model = Assert.IsAssignableFrom<IEnumerable<Product>>(viewResult.ViewData.Model);

            Assert.Equal(2, model.Count());
        }

        [Fact]
        public async Task Create_ReturnsBadRequestResult_WhenModelStateIsInvalid()
        {
            // Arrange
            var mockProductService = new Mock<IProductService>();

            var controller = new ProductController(mockProductService.Object);

            controller.ModelState.AddModelError("Name", "Required");

            var product = new Product { ProductID = 3 };

            // Act
            var result = await controller.Create(product);

            // Assert
            var badRequestResult = Assert.IsType<BadRequestObjectResult>(result);

            Assert.IsType<SerializableError>(badRequestResult.Value);
        }

        [Fact]
        public async Task Create_ReturnsARedirectAndAddProduct_WhenModelStateIsValid()
        {
            // Arrange
            var mockProductService = new Mock<IProductService>();

            var controller = new ProductController(mockProductService.Object);

            var product = new Product
            {
                ProductID = 3,
                Name = "Nodejs",
                Count = 11,
                Description = "this is node js"
            };

            // Act
            var result = await controller.Create(product);

            // Assert
            var redirectToActionResult = Assert.IsType<RedirectToActionResult>(result);
            Assert.Null(redirectToActionResult.ControllerName);
            Assert.Equal("List", redirectToActionResult.ActionName);
            mockProductService.Verify();
        }

        [Fact]
        public async Task GetProductById_ReturnsRedirectToIndexHome_WhenIdIsNull()
        {
            // Arrange
            var mockProductService = new Mock<IProductService>();

            var controller = new ProductController(mockProductService.Object);

            // Act
            var result = await controller.GetProductById(id: null);

            // Assert
            var redirectToActionResult =Assert.IsType<RedirectToActionResult>(result);
            Assert.Equal("Home", redirectToActionResult.ControllerName);
            Assert.Equal("Index", redirectToActionResult.ActionName);
        }

        [Fact]
        public async Task GetProductById_ContentWithProductNotFound_WhenIdNotFound()
        {
            // Arrange

            var mockProductService = new Mock<IProductService>();

            var controller = new ProductController(mockProductService.Object);

            int productId = 999;

            // Act
            var result = await controller.GetProductById(productId);

            // Assert
            var contentResult = Assert.IsType<ContentResult>(result);
            Assert.Equal("Product not found.", contentResult.Content);
        }

        [Fact]
        public async Task GetProductById_ViewResultWithProduct_WhenIdFound()
        {
            // Arrange

            var mockProductService = new Mock<IProductService>();

            var controller = new ProductController(mockProductService.Object);

            int productId = 1;

            // Act
            var result = await controller.GetProductById(productId);

            // Assert
            var viewResult = Assert.IsType<ViewResult>(result);

            var model = Assert.IsType<Product>(viewResult.ViewData.Model);
            Assert.Equal("F#", model.Name);
            Assert.Equal(11, model.ProductID);
        }
    }
}
